<?php

namespace App\Http\Controllers;

use App\Models\Candidature;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class CandidatureController extends Controller
{
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'fullName' => 'required|string|max:255',
            'phone' => 'required|string|max:50',
            'availability' => 'required|string|max:255',
            'educationLevel' => 'required|string',
            'experienceYears' => 'required|integer|min:0',
            'speciality' => 'required|string',
            'careerWish' => 'required|string',
            'expectations' => 'required|string',
            'cv' => 'required|file|mimes:pdf,doc,docx|max:10240',
            'rgpdConsent' => 'required|accepted',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // Stocker le CV
        $cvPath = null;
        if ($request->hasFile('cv')) {
            $cvFile = $request->file('cv');
            $cvPath = $cvFile->store('cvs', 'public');
        }

        // Créer la candidature
        $candidature = Candidature::create([
            'email' => $request->email,
            'full_name' => $request->fullName,
            'phone' => $request->phone,
            'availability' => $request->availability,
            'education_level' => $request->educationLevel,
            'experience_years' => $request->experienceYears,
            'speciality' => $request->speciality,
            'career_wish' => $request->careerWish,
            'expectations' => $request->expectations,
            'cv_path' => $cvPath,
            'rgpd_consent' => true,
        ]);

        // Envoyer l'email automatiquement
        try {
            $toEmail = env('MAIL_TO_ADDRESS', 'yessinegargouri@marqenti.tn');
            
            Mail::send('emails.nouvelle-candidature', ['candidature' => $candidature], function ($message) use ($candidature, $toEmail) {
                $message->to($toEmail)
                    ->subject('Nouvelle candidature - ' . $candidature->full_name);
                
                // Attacher le CV
                if ($candidature->cv_path) {
                    $message->attach(storage_path('app/public/' . $candidature->cv_path));
                }
            });
        } catch (\Exception $e) {
            \Log::error('Erreur envoi email: ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Candidature soumise avec succès',
            'data' => $candidature
        ], 201);
    }

    public function index()
    {
        $candidatures = Candidature::orderBy('created_at', 'desc')->get();
        return response()->json($candidatures);
    }

    public function show($id)
    {
        $candidature = Candidature::findOrFail($id);
        return response()->json($candidature);
    }

    public function markAsRead($id)
    {
        $candidature = Candidature::findOrFail($id);
        $candidature->is_read = true;
        $candidature->save();

        return response()->json([
            'success' => true,
            'message' => 'Candidature marquée comme lue'
        ]);
    }

    public function destroy($id)
    {
        $candidature = Candidature::findOrFail($id);
        
        // Supprimer le CV
        if ($candidature->cv_path) {
            Storage::disk('public')->delete($candidature->cv_path);
        }
        
        $candidature->delete();

        return response()->json([
            'success' => true,
            'message' => 'Candidature supprimée'
        ]);
    }

    public function downloadCv($id)
    {
        $candidature = Candidature::findOrFail($id);
        
        if (!$candidature->cv_path) {
            return response()->json(['error' => 'CV non trouvé'], 404);
        }

        $filePath = storage_path('app/public/' . $candidature->cv_path);
        
        if (!file_exists($filePath)) {
            return response()->json(['error' => 'Fichier non trouvé'], 404);
        }

        return response()->download($filePath);
    }
}
